/*
 *  Copyright 2019-2020 Zheng Jie
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package me.zhengjie.modules.system.repository;

import me.zhengjie.modules.system.domain.User;
import org.springframework.data.jpa.repository.JpaRepository;
import org.springframework.data.jpa.repository.JpaSpecificationExecutor;
import org.springframework.data.jpa.repository.Modifying;
import org.springframework.data.jpa.repository.Query;
import java.util.Date;
import java.util.List;
import java.util.Set;

/**
 * @author Zheng Jie
 * @date 2018-11-22
 */
public interface UserRepository extends JpaRepository<User, Long>, JpaSpecificationExecutor<User> {

    /**
     * 根据用户名查询
     * @param username 用户名
     * @return /
     */
    User findByUsername(String username);

    /**
     * 根据邮箱查询
     * @param email 邮箱
     * @return /
     */
    User findByEmail(String email);

    /**
     * 根据手机号查询
     * @param phone 手机号
     * @return /
     */
    User findByPhone(String phone);

    /**
     * 修改密码
     * @param username 用户名
     * @param pass 密码
     * @param lastPasswordResetTime /
     */
    @Modifying
    @Query(value = "update sjgx_sys_user set password = ?2 , pwd_reset_time = ?3 where username = ?1",nativeQuery = true)
    void updatePass(String username, String pass, Date lastPasswordResetTime);

    /**
     * 修改邮箱
     * @param username 用户名
     * @param email 邮箱
     */
    @Modifying
    @Query(value = "update sjgx_sys_user set email = ?2 where username = ?1",nativeQuery = true)
    void updateEmail(String username, String email);

    /**
     * 根据角色查询用户
     * @param roleId /
     * @return /
     */
    @Query(value = "SELECT U.* FROM sjgx_sys_user U, SJGX_SYS_USERS_ROLES R WHERE" +
            " U.USER_ID = R.USER_ID AND R.ROLE_ID = ?1", nativeQuery = true)
    List<User> findByRoleId(Long roleId);

    /**
     * 根据角色中的部门查询
     * @param deptId /
     * @return /
     */
    @Query(value = "SELECT U.* FROM sjgx_sys_user U, SJGX_SYS_USERS_ROLES R, SJGX_SYS_ROLES_DEPTS D WHERE " +
            "U.USER_ID = R.USER_ID AND R.ROLE_ID = D.ROLE_ID AND D.DEPT_ID = ?1 GROUP BY U.USER_ID", nativeQuery = true)
    List<User> findByRoleDeptId(Long deptId);

    /**
     * 根据菜单查询
     * @param id 菜单ID
     * @return /
     * //todo-sn 2021/9/26 11:00 Ash: 删掉"GROUP BY U.USER_ID"，不知道为啥要聚合
     */
    @Query(value = "SELECT U.* FROM sjgx_sys_user U, SJGX_SYS_USERS_ROLES UR, SJGX_SYS_ROLES_MENUS RM WHERE\n" +
            "U.USER_ID = UR.USER_ID AND UR.ROLE_ID = RM.ROLE_ID AND RM.MENU_ID = ?1 ", nativeQuery = true)
    List<User> findByMenuId(Long id);

    /**
     * 根据Id删除
     * @param ids /
     */
    void deleteAllByIdIn(Set<Long> ids);

    /**
     * 根据岗位查询
     * @param ids /
     * @return /
     */
    @Query(value = "SELECT COUNT(1) FROM sjgx_sys_user U, SJGX_SYS_USERS_JOBS J WHERE U.USER_ID = J.USER_ID AND J.JOB_ID IN ?1", nativeQuery = true)
    int countByJobs(Set<Long> ids);

    /**
     * 根据部门查询
     * @param deptIds /
     * @return /
     */
    @Query(value = "SELECT COUNT(1) FROM sjgx_sys_user U WHERE U.DEPT_ID IN ?1", nativeQuery = true)
    int countByDepts(Set<Long> deptIds);

    /**
     * 根据角色查询
     * @param ids /
     * @return /
     */
    @Query(value = "SELECT COUNT(1) FROM sjgx_sys_user U, SJGX_SYS_USERS_ROLES R WHERE " +
            "U.USER_ID = R.USER_ID AND R.ROLE_ID IN ?1", nativeQuery = true)
    int countByRoles(Set<Long> ids);
}
